<?php

namespace App\Http\Controllers;

use App\Models\Familia;
use App\Models\MiembroFamilia;
use Illuminate\Http\Request;

class FamiliaController extends Controller
{
    public function index(Request $request)
    {
        $query = Familia::with('miembros');

        $filters = [];
        if ($request->filled('email')) {
            $filters[] = ['email', $request->input('email')];
        }
        if ($request->filled('identificador')) {
            $filters[] = ['identificador', $request->input('identificador')];
        }
        if ($request->filled('telefono')) {
            $filters[] = ['telefono', $request->input('telefono')];
        }
        if ($request->filled('address')) {
            $filters[] = ['address', $request->input('address')];
        }

        if (!empty($filters)) {
            $query->where(function ($q) use ($filters) {
                foreach ($filters as $index => [$column, $value]) {
                    if ($index === 0) {
                        $q->where($column, 'like', '%' . $value . '%');
                    } else {
                        $q->orWhere($column, 'like', '%' . $value . '%');
                    }
                }
            });
        }

        if ($request->filled('rol_familiar')) {
            $rol = $request->input('rol_familiar');
            if (!in_array(strtolower($rol), ['todos'])) {
                $query->whereHas('miembros', function ($q) use ($rol) {
                    $q->where('rol_familiar', 'like', '%' . $rol . '%');
                });
            }
        }

        if ($request->has('per_page')) {
            return $query->paginate($request->input('per_page'));
        }

        return $query->get();
    }

    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string',
            'apellidos' => 'nullable|string',
            'telefono' => 'required|string',
            'address' => 'nullable|string',
            'email' => 'required|email',
            'foto' => 'nullable|string', // Base64
        ]);

        $data = $request->all();

        // Procesar foto del titular
        if ($request->has('foto')) {
            $data['foto'] = $this->saveBase64Image($request->foto, 'familias');
        }

        // El identificador se genera automáticamente en el modelo
        $familia = Familia::create($data);

        // Si se envían miembros en el mismo request
        if ($request->has('miembros') && is_array($request->miembros)) {
            foreach ($request->miembros as $miembroData) {
                if (isset($miembroData['foto'])) {
                    $miembroData['foto'] = $this->saveBase64Image($miembroData['foto'], 'miembros');
                }
                $familia->miembros()->create([
                    'nombre' => $miembroData['nombre'] ?? '',
                    'rol_familiar' => $miembroData['rol_familiar'] ?? '',
                    'sexo' => $miembroData['sexo'] ?? null,
                    'foto' => $miembroData['foto'] ?? null,
                ]);
            }
        }

        return response()->json([
            'message' => 'Familia creada exitosamente',
            'familia' => $familia->load('miembros')
        ], 201);
    }

    public function show($id)
    {
        return Familia::with('miembros')->findOrFail($id);
    }

    public function update(Request $request, $id)
    {
        $familia = Familia::findOrFail($id);

        $request->validate([
            'nombre' => 'sometimes|string',
            'apellidos' => 'nullable|string',
            'telefono' => 'sometimes|string',
            'email' => 'sometimes|email',
            'foto' => 'nullable|string',
            'miembros' => 'nullable|array',
            'miembros.*.id' => 'sometimes|exists:miembro_familias,id',
            'miembros.*.nombre' => 'sometimes|string',
            'miembros.*.rol_familiar' => 'sometimes|string',
            'miembros.*.sexo' => 'nullable|string',
            'miembros.*.foto' => 'nullable|string',
        ]);
        
        $data = $request->only(['nombre', 'apellidos', 'telefono', 'address', 'email']);

        if ($request->has('foto')) {
            $data['foto'] = $this->saveBase64Image($request->foto, 'familias');
        }

        if (!empty($data)) {
            $familia->update($data);
        }

        if ($request->has('miembros') && is_array($request->miembros)) {
            foreach ($request->miembros as $miembroData) {
                if (isset($miembroData['id'])) {
                    $miembro = $familia->miembros()->where('id', $miembroData['id'])->first();
                    if ($miembro) {
                        if (isset($miembroData['foto'])) {
                            $miembroData['foto'] = $this->saveBase64Image($miembroData['foto'], 'miembros');
                        }
                        $miembro->update([
                            'nombre' => $miembroData['nombre'] ?? $miembro->nombre,
                            'rol_familiar' => $miembroData['rol_familiar'] ?? $miembro->rol_familiar,
                            'sexo' => $miembroData['sexo'] ?? $miembro->sexo,
                            'foto' => $miembroData['foto'] ?? $miembro->foto,
                        ]);
                    }
                } else {
                    if (isset($miembroData['foto'])) {
                        $miembroData['foto'] = $this->saveBase64Image($miembroData['foto'], 'miembros');
                    }
                    $familia->miembros()->create([
                        'nombre' => $miembroData['nombre'] ?? '',
                        'rol_familiar' => $miembroData['rol_familiar'] ?? '',
                        'sexo' => $miembroData['sexo'] ?? null,
                        'foto' => $miembroData['foto'] ?? null,
                    ]);
                }
            }
        }
        
        return response()->json([
            'message' => 'Familia actualizada',
            'familia' => $familia->load('miembros')
        ]);
    }

    public function destroy($id)
    {
        $familia = Familia::findOrFail($id);
        $familia->delete(); // Elimina también miembros por el cascade
        
        return response()->json(['message' => 'Familia eliminada']);
    }

    public function stats()
    {
        $totalFamilias = Familia::count();
        $totalMiembros = MiembroFamilia::count();

        $inicioMes = now()->startOfMonth();
        $finMes = now()->endOfMonth();

        $miembrosMesActual = MiembroFamilia::whereBetween('created_at', [$inicioMes, $finMes])->count();

        return response()->json([
            'total_familias' => $totalFamilias,
            'total_miembros' => $totalMiembros,
            'miembros_mes_actual' => $miembrosMesActual,
        ]);
    }
}
